/******************************************************************************************************
*  TEXAS INSTRUMENTS INC.,                                                                            *
*  MSP430 APPLICATIONS.                                                                               *
*  Copyright Texas Instruments Inc, 2004                                                              *
 *****************************************************************************************************
*  This header file is for the MSP430_RF application.                                                 *
*  Compiler: IAR Workbench 3.0 or MSP430-GCC                                                                       *
 *****************************************************************************************************/

#ifndef HAL_H
#define HAL_H


/***********************************************************
	FAST SPI: Low level functions
***********************************************************/
#define	SPI_TxREADY()	while (!(IFG1 & UTXIFG0))
#define	SPI_WAITFOREOTx()	while (!(U0TCTL & TXEPT))
#define	SPI_WAITFOREORx()	while (!(IFG1 & URXIFG0))

#define FASTSPI_TX(x)\
	do {\
		SPI_TxREADY();\
		U0TXBUF = x;\
		SPI_WAITFOREOTx();\
	} while(0)

#define FASTSPI_RX(x)\
    do {\
		SPI_TxREADY();\
		U0TXBUF = 0;\
		SPI_WAITFOREOTx();\
		SPI_WAITFOREORx();\
		x = U0RXBUF;\
    } while(0)

#define FASTSPI_RX_GARBAGE()\
	do {\
    		SPI_TxREADY();\
		U0TXBUF = 0;\
		SPI_WAITFOREOTx();\
		SPI_WAITFOREORx();\
		U0RXBUF;\
	} while(0)

#define FASTSPI_TX_MANY(p,c)\
	do {\
		for (spiCnt = 0; spiCnt < (c); spiCnt++) {\
			FASTSPI_TX(((BYTE*)(p))[spiCnt]);\
		}\
	} while(0)


#define FASTSPI_RX_WORD(x)\
	 do {\
		SPI_TxREADY();\
		U0TXBUF = 0;\
		SPI_WAITFOREOTx();\
		SPI_WAITFOREORx();\
		x = U0RXBUF << 8;\
		SPI_TxREADY();\
		U0TXBUF = 0;\
		SPI_WAITFOREOTx();\
		SPI_WAITFOREORx();\
		x |= U0RXBUF;\
    } while (0)

#define FASTSPI_TX_ADDR(a)\
	 do {\
		SPI_TxREADY();\
		U0TXBUF = a;\
		SPI_WAITFOREOTx();\
	 } while (0)

#define FASTSPI_RX_ADDR(a)\
	 do {\
		SPI_TxREADY();\
		U0TXBUF = (a) | 0x40;\
		SPI_WAITFOREOTx();\
	 } while (0)



/***********************************************************
	FAST SPI: Register access
***********************************************************/
// 	  s = command strobe
// 	  a = register address
// 	  v = register value

#define FASTSPI_STROBE(s) \
    do {\
		  SPI_ENABLE();\
		  FASTSPI_TX_ADDR(s);\
		  SPI_DISABLE();\
    } while (0)

#define FASTSPI_SETREG(a,v)\
	 do {\
		  SPI_ENABLE();\
		  FASTSPI_TX_ADDR(a);\
		  FASTSPI_TX((BYTE) ((v) >> 8));\
		  FASTSPI_TX((BYTE) (v));\
		  SPI_DISABLE();\
	 } while (0)


#define FASTSPI_GETREG(a,v)\
	 do {\
		  SPI_ENABLE();\
		  FASTSPI_RX_ADDR(a);\
		  v= (BYTE)U0RXBUF;\
		  FASTSPI_RX_WORD(v);\
		  halWait(1);\
		  SPI_DISABLE();\
	 } while (0)

// Updates the SPI status byte

#define FASTSPI_UPD_STATUS(s)\
	 do {\
		  SPI_ENABLE();\
		  U0TXBUF = CC2420_SNOP;\
		  SPI_WAITFOREOTx();\
		  s = U0RXBUF;\
		  SPI_DISABLE();\
	 } while (0)

/***********************************************************
	FAST SPI: FIFO Access
***********************************************************/
// 	  p = pointer to the byte array to be read/written
// 	  c = the number of bytes to read/write
// 	  b = single data byte

#define FASTSPI_WRITE_FIFO(p,c)\
	do {\
		SPI_ENABLE();\
		FASTSPI_TX_ADDR(CC2420_TXFIFO);\
		for (i = 0; i < (c); i++) {\
		    FASTSPI_TX(((BYTE*)(p))[i]);\
		}\
		SPI_DISABLE();\
    } while (0)

#define FASTSPI_WRITE_FIFO_NOCE(p,c)\
	do {\
		SPI_ENABLE();\
		FASTSPI_TX_ADDR(CC2420_TXFIFO);\
		for (spiCnt = 0; spiCnt < (c); spiCnt++) {\
		    FASTSPI_TX(((BYTE*)(p))[spiCnt]);\
		SPI_DISABLE();\
		}\
    } while (0)

#define FASTSPI_READ_FIFO_BYTE(b)\
	 do {\
		  SPI_ENABLE();\
		  FASTSPI_RX_ADDR(CC2420_RXFIFO);\
		  U0RXBUF;\
		  FASTSPI_RX(b);\
  		  halWait(1);\
		  SPI_DISABLE();\
	 } while (0)


#define FASTSPI_READ_FIFO_NO_WAIT(p,c)\
	 do {\
		  SPI_ENABLE();\
		  FASTSPI_RX_ADDR(CC2420_RXFIFO);\
		  U0RXBUF;\
		  for (spiCnt = 0; spiCnt < (c); spiCnt++) {\
				FASTSPI_RX(((BYTE*)(p))[spiCnt]);\
		  }\
		  halWait(1);\
		  SPI_DISABLE();\
	 } while (0)



#define FASTSPI_READ_FIFO_GARBAGE(c)\
	 do {\
		  SPI_ENABLE();\
		  FASTSPI_RX_ADDR(CC2420_RXFIFO);\
		  U0RXBUF;\
		  for (spiCnt = 0; spiCnt < (c); spiCnt++) {\
				FASTSPI_RX_GARBAGE();\
		  }\
  		  halWait(1);\
		  SPI_DISABLE();\
	 } while (0)



/***********************************************************
	FAST SPI: CC2420 RAM access (big or little-endian order)
***********************************************************/
//  FAST SPI: CC2420 RAM access (big or little-endian order)
// 	  p = pointer to the variable to be written
// 	  a = the CC2420 RAM address
// 	  c = the number of bytes to write
// 	  n = counter variable which is used in for/while loops (UINT8)
//
//  Example of usage:
// 	  UINT8 n;
// 	  UINT16 shortAddress = 0xBEEF;
// 	  FASTSPI_WRITE_RAM_LE(&shortAddress, CC2420RAM_SHORTADDR, 2);


#define FASTSPI_WRITE_RAM_LE(p,a,c,n)\
	 do {\
		  SPI_ENABLE();\
		  FASTSPI_TX(0x80 | (a & 0x7F));\
		  FASTSPI_TX((a >> 1) & 0xC0);\
		  for (n = 0; n < (c); n++) {\
				FASTSPI_TX(((BYTE*)(p))[n]);\
		  }\
		  SPI_DISABLE();\
	 } while (0)

//-------------------------------------------------------------------------------------------------------
//  void halWait(UINT16 timeout)
//
//  DESCRIPTION:
//      Runs an idle loop for [timeout] microseconds.
//
//  ARGUMENTS:
//      UINT16 timeout
//          The timeout in microseconds
//-------------------------------------------------------------------------------------------------------
void halWait(UINT16 timeout);


/*******************************************************************************************************
 *******************************************************************************************************
 **************************              SIMPLE CC2420 FUNCTIONS              **************************
 *******************************************************************************************************
 *******************************************************************************************************/

//-------------------------------------------------------------------------------------------------------
//  Example of usage: Starts RX on channel 14 after reset
//      FASTSPI_RESET_CC2420();
//      FASTSPI_STROBE(CC2420_SXOSCON);
//      halRfSetChannel(14);
//      ... other registers can for instance be initialized here ...
//      halRfWaitForCrystalOscillator();
//      ... RAM access can be done here, since the crystal oscillator must be on and stable ...
//      FASTSPI_STROBE(CC2420_SRXON);
//-------------------------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------------------------
//  void rfWaitForCrystalOscillator(void)
//
//  DESCRIPTION:
//      Waits for the crystal oscillator to become stable. The flag is polled via the SPI status byte.
//
//      Note that this function will lock up if the SXOSCON command strobe has not been given before the
//      function call. Also note that global interrupts will always be enabled when this function
//      returns.
//-------------------------------------------------------------------------------------------------------
void halRfWaitForCrystalOscillator(void);


//-------------------------------------------------------------------------------------------------------
//  void halRfSetChannel(UINT8 Channel)
//
//  DESCRIPTION:
//      Programs CC2420 for a given IEEE 802.15.4 channel.
//      Note that SRXON, STXON or STXONCCA must be run for the new channel selection to take full effect.
//
//  PARAMETERS:
//      UINT8 channel
//          The channel number (11-26)
//-------------------------------------------------------------------------------------------------------
void halRfSetChannel(UINT8 channel);	 

void halSpiInit(void);

#endif
